% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mcgraph.R
\name{mcg.roc}
\alias{mcg.roc}
\title{Get common metrics for a given network classifier, as well as the Receiver Operating Characteristic (ROC) or precision-recall (PR) curve.}
\usage{
mcg.roc(d, G, thresholds, plot = "ROC", fun = mcg.lvs, ...)
}
\arguments{
\item{d}{Data frame with variables as columns and data values as rows.}

\item{G}{Adjacency matrix of the true network.}

\item{thresholds}{Vector of classification thresholds in increasing order.}

\item{plot}{Either 'ROC' for the classical ROC curve (sensitivity vs. 1 - FPR) or 'PRC' for the precision-recall curve. Default: 'ROC'}

\item{fun}{The name of the classifier function, either unquoted or with quotation marks. Default: mcg.lvs}

\item{...}{additional arguments for 'fun'}
}
\value{
a list containing values for several metrics (see Details).
}
\description{
\code{mcg.roc} can be used to evaluate the quality of network estimates.
             The function returns common evaluation metrics for given classification instances for a single network classifier and allows
             for plots of ROC and precision-recall curves.
}
\details{
If the true network given by `G` is directed, it will be transformed to an undirected graph. Negative or weighted edges
         are not considered and are converted to 1-entries. The diagonal of the adjacency matrix is excluded. If the argument `plot='R'` or
         `plot='PRC'` is set, the ROC curve or the precision-recall curve will be plotted, respectively.
         The argument 'fun' either takes a classifier function from the `mcgraph` package, e.g.:
         \describe{
         \item{\code{mcg.lvs}}{forward variable selection in linear models}
         \item{\code{mcg.rpart}}{variable selection using decision trees}
         \item{\code{mcg.ct}}{pruning based on hard correlation thresholding}
         }
         or a costum function either given by its function name or in the form of \code{function(x) \{ \}}. The given function
         will then be matched and called, after evaluating possible additional arguments given for \code{...}.
         There are three requirements for a given function
         \enumerate{
             \item{it must take a data object as a first argument with variables as columns and values as rows}
             \item{there must be an argument named `thr` for a classification threshold taking a single numeric value}
             \item{the given function must return an adjacency matrix}
         }
         Functions from other packages might be given in the form of wrappers to satisfy the above conditions.

         \code{mcg.roc} returns a list containing the given classification thresholds \code{thr} and following metrics for each instance:
         The accuracy \code{acc}, sensitivity \code{sen}, specificity \code{spe}, precision \code{pre}, false-postive rate \code{fpr},
         Matthews Correlation Coefficient [MCC] \code{mcc}, the normed MCC \code{norm_mcc} calculated by \code{(MCC + 1) / 2} and the F1-score \code{f1}.
         Based on the trapezoid rule, the area under the ROC curve \code{auc} and the area under the precision-recall curve \code{pr.auc} are returned.
         Instances, which cannot be calculated because they are mathematically not defined, are set to \code{NA} by convention.

         Note, that the raw results are returned in order of the given classification instances. For internal calculation of the area under
         the curve and plotting, additional start- and endpoints are be included and the values are sorted in descending order
         after excluding \code{NA} entries. For convenient reuse for own plots, those values are also returned by the nested list
         entries \code{roc} and \code{prc}, respectively.
}
\examples{
# Build lion star sign network
L <- mcg.constellation(name="lion")

# Create Monte-Carlo data for given network
set.seed(1234)
d <- as.data.frame(t(mcg.graph2data(L)))

# Create & plot ROC curves using forward linear variable selection
mcg.roc(d=d, G=L, thresholds=seq(0, 1, length.out=11), fun=mcg.lvs)

# Variable selection using decision tree
mcg.roc(d=d, G=L, thresholds=seq(0, 1, length.out=11), fun=mcg.rpart)

# Hard correlation thresholding using R-square values
mcg.roc(d=d, G=L, thresholds=seq(0, 1, length.out=11), fun=mcg.ct)

# Example for a costum function to perform hard correlation thresholding
hct <- function(d, thr=0.5) {
    # Init adjacency matrix
    A <- matrix(0, nrow=ncol(d), ncol=ncol(d))
    # Calculate correlation matrix
    cor_mt <- cor(d)
    # Exclude diagonal of correlation matrix
    diag(cor_mt) <- 0
    # Simple hard correlation thresholding
    A[which(cor_mt >= thr)] <- 1
    return(A)
}
# Get metrics and plot precision-recall curve
mcg.roc(d=d, G=L, thresholds=c(0.2, 0.4, 0.5), plot="prc", fun=hct)

}
\author{
Masiar Novine <email: masiar.novine@gmail.com>
}
